<?php

namespace App\Services\Resource;

use App\Enums\Resource\AudienceType;
use App\Http\Resources\Academic\SubjectResource;
use App\Jobs\Notifications\Resource\SendBatchLearningMaterialNotification;
use App\Models\Academic\BatchSubjectRecord;
use App\Models\Academic\Subject;
use App\Models\Audience;
use App\Models\Employee\Employee;
use App\Models\Resource\LearningMaterial;
use App\Support\HasAudience;
use Illuminate\Http\Request;

class LearningMaterialService
{
    use HasAudience;

    public function preRequisite(Request $request): array
    {
        $subjects = SubjectResource::collection(Subject::query()
            ->byPeriod()
            ->get());

        $audienceTypes = AudienceType::getOptions();

        return compact('subjects', 'audienceTypes');
    }

    public function create(Request $request): LearningMaterial
    {
        \DB::beginTransaction();

        $learningMaterial = LearningMaterial::forceCreate($this->formatParams($request));

        if ($request->audience_type == 'batch_wise') {
            $this->updateBatchSubjectRecords($request, $learningMaterial);
        } else {
            $this->updateStudentRecords($request, $learningMaterial);
        }

        $learningMaterial->addMedia($request);

        \DB::commit();

        SendBatchLearningMaterialNotification::dispatch([
            'learning_material_id' => $learningMaterial->id,
            'sender_user_id' => auth()->id(),
            'team_id' => auth()->user()->current_team_id,
        ]);

        return $learningMaterial;
    }

    private function formatParams(Request $request, ?LearningMaterial $learningMaterial = null): array
    {
        $formatted = [
            'title' => $request->title,
            'description' => clean($request->description),
            'published_at' => now()->toDateTimeString(),
        ];

        $config = $learningMaterial?->config ?? [];
        $config['audience_type'] = $request->audience_type;
        $formatted['config'] = $config;

        if (! $learningMaterial) {
            $formatted['period_id'] = auth()->user()->current_period_id;
            $formatted['employee_id'] = Employee::auth()->first()?->id;
        }

        return $formatted;
    }

    private function updateBatchSubjectRecords(Request $request, LearningMaterial $learningMaterial)
    {
        if ($request->audience_type == 'student_wise') {
            BatchSubjectRecord::query()
                ->whereModelType($learningMaterial->getMorphClass())
                ->whereModelId($learningMaterial->id)
                ->delete();

            return;
        }

        $usedIds = [];
        foreach ($request->batch_ids as $batchId) {
            $usedIds[] = [
                'batch_id' => $batchId,
                'subject_id' => $request->subject_id,
            ];
            BatchSubjectRecord::firstOrCreate([
                'model_type' => $learningMaterial->getMorphClass(),
                'model_id' => $learningMaterial->id,
                'batch_id' => $batchId,
                'subject_id' => $request->subject_id,
            ]);
        }
        $records = BatchSubjectRecord::query()
            ->whereModelType($learningMaterial->getMorphClass())
            ->whereModelId($learningMaterial->id)
            ->get();
        $usedIds = collect($usedIds);
        foreach ($records as $record) {
            if (! $usedIds->where('batch_id', $record->batch_id)->where('subject_id', $record->subject_id)->count()) {
                $record->delete();
            }
        }
    }

    private function updateStudentRecords(Request $request, LearningMaterial $learningMaterial)
    {
        if ($request->audience_type == 'batch_wise') {
            Audience::query()
                ->whereShareableType($learningMaterial->getMorphClass())
                ->whereShareableId($learningMaterial->id)
                ->whereAudienceableType('Student')
                ->delete();

            return;
        }

        $usedIds = [];
        foreach ($request->student_ids as $studentId) {
            $audience = Audience::query()
                ->firstOrCreate([
                    'shareable_type' => $learningMaterial->getMorphClass(),
                    'shareable_id' => $learningMaterial->id,
                    'audienceable_type' => 'Student',
                    'audienceable_id' => $studentId,
                ]);

            $usedIds[] = $studentId;
        }

        Audience::query()
            ->whereShareableType($learningMaterial->getMorphClass())
            ->whereShareableId($learningMaterial->id)
            ->whereAudienceableType('Student')
            ->whereNotIn('audienceable_id', $usedIds)
            ->delete();
    }

    public function update(Request $request, LearningMaterial $learningMaterial): void
    {
        \DB::beginTransaction();

        $learningMaterial->forceFill($this->formatParams($request, $learningMaterial))->save();

        $this->updateBatchSubjectRecords($request, $learningMaterial);

        $this->updateStudentRecords($request, $learningMaterial);

        $learningMaterial->updateMedia($request);

        \DB::commit();
    }

    public function deletable(LearningMaterial $learningMaterial): void
    {
        //
    }
}
