<?php

namespace App\Policies;

use App\Models\Guardian;
use App\Models\User;
use Illuminate\Auth\Access\HandlesAuthorization;

class GuardianPolicy
{
    use HandlesAuthorization;

    private function validateTeam(User $user, Guardian $guardian)
    {
        return $guardian->contact->team_id == $user->current_team_id;
    }

    /**
     * Determine whether the user can request for pre-requisites.
     *
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function preRequisite(User $user)
    {
        return $user->canAny(['guardian:create', 'guardian:edit']);
    }

    /**
     * Determine whether the user can view any models.
     *
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function viewAny(User $user)
    {
        return $user->can('guardian:read');
    }

    /**
     * Determine whether the user can view the model.
     *
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function view(User $user, Guardian $guardian)
    {
        if (! $this->validateTeam($user, $guardian)) {
            return false;
        }

        return $user->can('guardian:read');
    }

    /**
     * Determine whether the user can create models.
     *
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function create(User $user)
    {
        return $user->can('guardian:create');
    }

    /**
     * Determine whether the user can update the model.
     *
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function update(User $user, Guardian $guardian)
    {
        if (! $this->validateTeam($user, $guardian)) {
            return false;
        }

        return $user->can('guardian:edit');
    }

    /**
     * Determine whether the user can update the model.
     *
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function photoUpdate(User $user, Guardian $guardian)
    {
        if (! $this->validateTeam($user, $guardian)) {
            return false;
        }

        if ($user->can('guardian:edit')) {
            return true;
        }

        if ($user->can('student:edit')) {
            return true;
        }

        return $user->can('student-record:manage');
    }

    /**
     * Determine whether the user can delete the model.
     *
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function delete(User $user, Guardian $guardian)
    {
        if (! $this->validateTeam($user, $guardian)) {
            return false;
        }

        return $user->can('guardian:delete');
    }

    /**
     * Determine whether the user can restore the model.
     *
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function restore(User $user, Guardian $guardian)
    {
        //
    }

    /**
     * Determine whether the user can permanently delete the model.
     *
     * @return \Illuminate\Auth\Access\Response|bool
     */
    public function forceDelete(User $user, Guardian $guardian)
    {
        //
    }
}
