<?php

namespace App\Http\Requests\Reception;

use App\Enums\OptionType;
use App\Enums\Reception\EnquiryStatus;
use App\Models\Academic\Course;
use App\Models\Academic\Period;
use App\Models\Academic\Program;
use App\Models\Option;
use App\Models\Reception\Enquiry;
use App\Models\Team;
use App\Rules\AlphaSpace;
use Illuminate\Foundation\Http\FormRequest;

class OnlineEnquiryRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return true;
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        $rules = [
            'team' => 'required|uuid',
            // 'program' => 'required|uuid',
            'period' => 'required|uuid',
            'course' => 'required|uuid',
            // 'batch' => 'nullable|uuid',
            // 'name' => ['required', 'min:2', 'max:100', new AlphaSpace],
            'first_name' => ['required', 'min:2', 'max:100', new AlphaSpace],
            'middle_name' => ['nullable', 'max:100', new AlphaSpace],
            'third_name' => ['nullable', 'max:100', new AlphaSpace],
            'last_name' => ['nullable', 'max:100', new AlphaSpace],
            'gender' => 'sometimes|required',
            'birth_date' => 'sometimes|required|date_format:Y-m-d',
            'contact_number' => 'required|min:10|max:20',
            'email' => 'required|email|max:100',
            'guardian_name' => 'nullable|max:100',
        ];

        return $rules;
    }

    public function withValidator($validator)
    {
        if (! $validator->passes()) {
            return;
        }

        $validator->after(function ($validator) {

            $team = Team::query()
                ->whereUuid($this->team)
                ->getOrFail(trans('validation.exists', ['attribute' => trans('team.team')]), 'team');

            // $program = Program::query()
            //     ->byTeam($team->id)
            //     ->whereUuid($this->program)
            //     ->getOrFail(trans('validation.exists', ['attribute' => trans('academic.program.program')]), 'program');

            // if (! $program->getConfig('enable_registration')) {
            //     $validator->errors()->add('program', trans('academic.program.registration_disabled_info'));
            // }

            $period = Period::query()
                ->whereTeamId($team->id)
                ->whereUuid($this->period)
                ->getOrFail(trans('validation.exists', ['attribute' => trans('academic.period.period')]), 'period');

            if (! $period->getConfig('enable_registration')) {
                $validator->errors()->add('period', trans('academic.period.registration_disabled_info'));
            }

            $course = Course::query()
                ->byPeriod($period->id)
                ->whereUuid($this->course)
                ->getOrFail(trans('validation.exists', ['attribute' => trans('academic.course.course')]), 'course');

            if (! $course->enable_registration) {
                $validator->errors()->add('course', trans('academic.course.registration_disabled_info'));
            }

            // if (! $course->batches->count()) {
            //     $validator->errors()->add('course', trans('reception.online_enquiry.no_batches_available'));
            // }

            $existingEnquiry = Enquiry::query()
                ->select('enquiries.*')
                ->where('name', $this->name)
                ->where(function ($q) {
                    $q->where('enquiries.contact_number', $this->contact_number)
                        ->orWhere('enquiries.email', $this->email);
                })
                ->first();

            if ($existingEnquiry && $existingEnquiry->status == EnquiryStatus::OPEN) {
                $validator->errors()->add('message', trans('reception.online_enquiry.pending_enquiry_exists'));
            }

            $defaultEnquiryStage = Option::query()
                ->byTeam($team->id)
                ->where('type', OptionType::ENQUIRY_STAGE)
                ->where('name', 'New Enquiry')
                ->first();

            $defaultEnquirySource = Option::query()
                ->byTeam($team->id)
                ->where('type', OptionType::ENQUIRY_SOURCE)
                ->where('name', 'Website')
                ->first();

            $defaultEnquiryType = Option::query()
                ->byTeam($team->id)
                ->where('type', OptionType::ENQUIRY_TYPE)
                ->where('name', 'Online Enquiry')
                ->first();

            $this->merge([
                'team_id' => $team->id,
                // 'program_id' => $program?->id,
                'period_id' => $period?->id,
                'course_id' => $course?->id,
                'enquiry_source_id' => $defaultEnquirySource?->id,
                'enquiry_stage_id' => $defaultEnquiryStage?->id,
                'enquiry_type_id' => $defaultEnquiryType?->id,
            ]);
        });
    }

    /**
     * Translate fields with user friendly name.
     *
     * @return array
     */
    public function attributes()
    {
        return [
            'team' => __('team.team'),
            'program' => __('academic.program.program'),
            'period' => __('academic.period.period'),
            'course' => __('academic.course.course'),
            'batch' => __('academic.batch.batch'),
            'name' => __('contact.props.name'),
            'first_name' => __('contact.props.first_name'),
            'middle_name' => __('contact.props.middle_name'),
            'third_name' => __('contact.props.third_name'),
            'last_name' => __('contact.props.last_name'),
            'gender' => __('contact.props.gender'),
            'birth_date' => __('contact.props.birth_date'),
            'contact_number' => __('contact.props.contact_number'),
            'email' => __('contact.props.email'),
        ];
    }
}
